<?php
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: giris.php");
    exit;
}

require 'config.php';

// Mevcut acil durumları çek
$stmt = $pdo->query("SELECT * FROM acil_durumlar ORDER BY timestamp DESC");
$locations = $stmt->fetchAll(PDO::FETCH_ASSOC);

// İstatistikler için verileri çek
$active_cases = $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE status != 'resolved'")->fetchColumn();
$high_priority = $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE priority = 3")->fetchColumn();
$team_count = $pdo->query("SELECT COUNT(*) FROM ekipler")->fetchColumn();

// Grafik verileri
$chart_data = [
    'deprem' => $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE emergency_type = 'deprem'")->fetchColumn(),
    'sel' => $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE emergency_type = 'sel'")->fetchColumn(),
    'yangin' => $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE emergency_type = 'yangin'")->fetchColumn(),
    'digery' => $pdo->query("SELECT COUNT(*) FROM acil_durumlar WHERE emergency_type NOT IN ('deprem', 'sel', 'yangin')")->fetchColumn()
];
?>
<!DOCTYPE html>
<html lang="tr">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
<title>İhbar Kontrol Sistemi</title>
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
<link rel="manifest" href="manifest.json">
<style>
  body {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    margin: 0; padding: 0;
  }
  h1 {
    background: #b30000;
    color: white;
    margin: 0;
    padding: 15px;
    text-align: center;
  }
  #map {
    height: 500px;
    margin-bottom: 20px;
  }
  table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
  }
  th, td {
    padding: 10px;
    border: 1px solid #ddd;
    text-align: center;
    font-size: 14px;
  }
  th {
    background-color: #f2f2f2;
    font-weight: bold;
  }
  tr:nth-child(even) {
    background-color: #f9f9f9;
  }
  tr:hover {
    background-color: #f1f1f1;
  }
  
  /* Acil durum türü etiketleri */
  .emergency-tag {
    display: inline-block;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: bold;
    color: white;
    min-width: 100px;
    text-align: center;
  }
  .deprem { background-color: #e74c3c; }
  .sel { background-color: #3498db; }
  .yangin { background-color: #f39c12; }
  .bina_yikilmasi { background-color: #9b59b6; }
  .trafik_kazasi { background-color: #1abc9c; }
  .medikal { background-color: #e67e22; }
  .digery { background-color: #7f8c8d; }

  /* Telefon numarası stili */
  .phone-number {
    font-family: monospace;
    direction: ltr;
    display: inline-block;
  }

  /* Loader kaplayıcı */
  #loader {
    position: fixed;
    top: 0; left: 0; right: 0; bottom: 0;
    background: #fff;
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 9999;
    overflow: hidden;
  }

  /* Logo resmi */
  #logoLoader {
    width: 150px;
    height: 150px;
    cursor: default;
    animation: logoPulse 2.5s ease-in-out forwards;
    position: relative;
    filter: drop-shadow(0 0 8px rgba(179, 0, 0, 0.7));
  }

  @keyframes logoPulse {
    0% {
      transform: scale(0.7);
      opacity: 0;
    }
    50% {
      transform: scale(1.2);
      opacity: 1;
    }
    100% {
      transform: scale(1);
      opacity: 1;
    }
  }

  /* Konum linkleri */
  .location-link {
    color: #0066cc;
    text-decoration: none;
    cursor: pointer;
  }
  .location-link:hover {
    text-decoration: underline;
  }

  /* Filtreleme alanı */
  .filters {
    margin: 15px;
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
  }
  .filters select, .filters button {
    padding: 8px 12px;
    border-radius: 4px;
    border: 1px solid #ddd;
  }

  /* İstatistik paneli */
  .stats-panel {
    background: #f5f5f5;
    padding: 15px;
    margin: 20px;
    border-radius: 8px;
  }
  .stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-bottom: 20px;
  }
  .stat-card {
    background: white;
    padding: 15px;
    border-radius: 5px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    text-align: center;
  }
  .stat-card h4 {
    margin-top: 0;
    color: #555;
  }
  .stat-card span {
    font-size: 24px;
    font-weight: bold;
    color: #b30000;
  }

  /* Öncelik durum renkleri */
  .priority-1 { background-color: #d4edda; }
  .priority-2 { background-color: #fff3cd; }
  .priority-3 { background-color: #f8d7da; }

  /* Durum renkleri */
  .status-pending { background-color: #fff3cd; }
  .status-assigned { background-color: #cce5ff; }
  .status-resolved { background-color: #d4edda; }

  /* Mobil uyumluluk */
  @media (max-width: 768px) {
    table {
      font-size: 12px;
    }
    th, td {
      padding: 5px;
    }
    .filters {
      flex-direction: column;
    }
  }
</style>
</head>
<body>
  <div id="loader">
    <img src="image/karkut.png" alt="KARKUT Logo" id="logoLoader" />
  </div>

<h1>Acil Durum Bildirim Sistemi Yönetici Paneli</h1>

<!-- Filtreleme ve Gruplama -->
<div class="filters">
    <select id="emergencyFilter">
        <option value="all">Tüm Acil Durumlar</option>
        <option value="deprem">Deprem</option>
        <option value="sel">Sel</option>
        <option value="yangin">Yangın</option>
        <option value="bina_yikilmasi">Bina Yıkılması</option>
        <option value="trafik_kazasi">Trafik Kazası</option>
        <option value="medikal">Medikal</option>
    </select>
    
    <select id="priorityFilter">
        <option value="all">Tüm Öncelikler</option>
        <option value="1">Düşük Öncelik</option>
        <option value="2">Orta Öncelik</option>
        <option value="3">Yüksek Öncelik</option>
    </select>
    
    <select id="statusFilter">
        <option value="all">Tüm Durumlar</option>
        <option value="pending">Beklemede</option>
        <option value="assigned">İşleme Alındı</option>
        <option value="resolved">Çözüldü</option>
    </select>
    
    <button id="groupBtn" class="btn btn-primary">
        <i class="bi bi-collection"></i> Yakın Konumları Grupla
    </button>
</div>

<!-- İstatistik Paneli -->
<div class="stats-panel">
    <h3>Operasyon İstatistikleri</h3>
    <div class="stats-grid">
        <div class="stat-card">
            <h4>Aktif Durumlar</h4>
            <span><?= $active_cases ?></span>
        </div>
        <div class="stat-card">
            <h4>Yüksek Öncelik</h4>
            <span><?= $high_priority ?></span>
        </div>
        <div class="stat-card">
            <h4>Ekipler</h4>
            <span><?= $team_count ?></span>
        </div>
    </div>
    <canvas id="incidentChart" width="400" height="200"></canvas>
</div>

<!-- Harita -->
<div id="map"></div>

<!-- Acil Durum Tablosu -->
<table>
  <thead>
    <tr>
      <th>ID</th>
      <th>Ad Soyad</th>
      <th>Telefon</th>
      <th>Konum</th>
      <th>Acil Durum Türü</th>
      <th>Öncelik</th>
      <th>Durum</th>
      <th>Ekip</th>
      <th>Zaman</th>
      <th>IP</th>
    </tr>
  </thead>
  <tbody>
    <?php foreach($locations as $loc): ?>
    <tr class="priority-<?= $loc['priority'] ?? 1 ?> status-<?= $loc['status'] ?? 'pending' ?>">
      <td><?= htmlspecialchars($loc['id']) ?></td>
      <td>
        <?= htmlspecialchars($loc['name'] ?? '') ?> <?= htmlspecialchars($loc['surname'] ?? '') ?>
      </td>
      <td>
        <span class="phone-number">
          <?= htmlspecialchars(formatPhoneNumber($loc['phone'] ?? '')) ?>
        </span>
      </td>
      <td>
        <a href="https://www.google.com/maps?q=<?= htmlspecialchars($loc['latitude']) ?>,<?= htmlspecialchars($loc['longitude']) ?>" 
           target="_blank" class="location-link">
          <?= htmlspecialchars(round($loc['latitude'], 5)) ?>, <?= htmlspecialchars(round($loc['longitude'], 5)) ?>
        </a>
      </td>
      <td>
        <span class="emergency-tag <?= htmlspecialchars($loc['emergency_type'] ?? 'digery') ?>">
          <?= htmlspecialchars(getEmergencyTypeLabel($loc['emergency_type'] ?? 'diğer')) ?>
        </span>
      </td>
      <td>
        <select class="priority-select" data-id="<?= $loc['id'] ?>">
            <option value="1" <?= ($loc['priority'] ?? 1) == 1 ? 'selected' : '' ?>>Düşük</option>
            <option value="2" <?= ($loc['priority'] ?? 1) == 2 ? 'selected' : '' ?>>Orta</option>
            <option value="3" <?= ($loc['priority'] ?? 1) == 3 ? 'selected' : '' ?>>Yüksek</option>
        </select>
      </td>
      <td>
        <select class="status-select" data-id="<?= $loc['id'] ?>">
            <option value="pending" <?= ($loc['status'] ?? 'pending') == 'pending' ? 'selected' : '' ?>>Beklemede</option>
            <option value="assigned" <?= ($loc['status'] ?? 'pending') == 'assigned' ? 'selected' : '' ?>>İşleme Alındı</option>
            <option value="resolved" <?= ($loc['status'] ?? 'pending') == 'resolved' ? 'selected' : '' ?>>Çözüldü</option>
        </select>
      </td>
      <td>
        <?php if ($loc['status'] == 'assigned'): ?>
            <span><?= htmlspecialchars(getTeamName($pdo, $loc['assigned_team'])) ?></span>
        <?php else: ?>
            <button class="assign-btn" data-id="<?= $loc['id'] ?>">Ekip Ata</button>
            <div class="team-assignment" style="display:none;">
                <select class="team-select">
                    <?php 
                    $teams = $pdo->query("SELECT * FROM ekipler")->fetchAll();
                    foreach($teams as $team): ?>
                    <option value="<?= $team['id'] ?>"><?= $team['name'] ?></option>
                    <?php endforeach; ?>
                </select>
                <button class="confirm-assign">Onayla</button>
            </div>
        <?php endif; ?>
      </td>
      <td><?= htmlspecialchars($loc['timestamp']) ?></td>
      <td><?= htmlspecialchars($loc['ip']) ?></td>
    </tr>
    <?php endforeach; ?>
  </tbody>
</table>

<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
  // Harita oluşturma
  var map = L.map('map').setView([40.995, 29.113], 12);
  L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
    maxZoom: 18
  }).addTo(map);

  // Marker'ları ekle
  <?php foreach($locations as $loc): ?>
    L.marker([<?= $loc['latitude'] ?>, <?= $loc['longitude'] ?>])
      .addTo(map)
      .on('click', function() {
        window.open(`https://www.google.com/maps?q=<?= $loc['latitude'] ?>,<?= $loc['longitude'] ?>`, '_blank');
      })
      .bindPopup(`
        <b>ID:</b> <?= $loc['id'] ?><br>
        <b>Ad Soyad:</b> <?= htmlspecialchars($loc['name'] ?? '') ?> <?= htmlspecialchars($loc['surname'] ?? '') ?><br>
        <b>Telefon:</b> <?= htmlspecialchars(formatPhoneNumber($loc['phone'] ?? '')) ?><br>
        <b>Tür:</b> <?= getEmergencyTypeLabel($loc['emergency_type'] ?? 'diğer') ?><br>
        <b>Öncelik:</b> ${getPriorityLabel(<?= $loc['priority'] ?? 1 ?>)}<br>
        <b>Durum:</b> ${getStatusLabel('<?= $loc['status'] ?? 'pending' ?>')}<br>
        <b>Zaman:</b> <?= $loc['timestamp'] ?><br>
        <a href="https://www.google.com/maps?q=<?= $loc['latitude'] ?>,<?= $loc['longitude'] ?>" target="_blank">Google Maps'te Aç</a>
      `);
  <?php endforeach; ?>

  // Logo animasyon kontrolü
  const logo = document.getElementById('logoLoader');
  const loader = document.getElementById('loader');

  setTimeout(() => {
    logo.style.opacity = '0';
    const explodeWrapper = document.createElement('div');
    explodeWrapper.id = 'logoExplode';
    for(let i = 0; i < 9; i++) {
      const piece = document.createElement('div');
      piece.className = 'piece';
      explodeWrapper.appendChild(piece);
    }
    loader.appendChild(explodeWrapper);
    explodeWrapper.style.opacity = '1';
    setTimeout(() => {
      loader.style.display = 'none';
    }, 1500);
  }, 3000);

  // Filtreleme fonksiyonları
  document.getElementById('emergencyFilter').addEventListener('change', filterTable);
  document.getElementById('priorityFilter').addEventListener('change', filterTable);
  document.getElementById('statusFilter').addEventListener('change', filterTable);

  function filterTable() {
    const type = document.getElementById('emergencyFilter').value;
    const priority = document.getElementById('priorityFilter').value;
    const status = document.getElementById('statusFilter').value;
    
    document.querySelectorAll('tbody tr').forEach(row => {
      const typeMatch = type === 'all' || row.querySelector('.emergency-tag').classList.contains(type);
      const priorityMatch = priority === 'all' || row.querySelector('.priority-select').value === priority;
      const statusMatch = status === 'all' || row.querySelector('.status-select').value === status;
      
      row.style.display = (typeMatch && priorityMatch && statusMatch) ? '' : 'none';
    });
  }

  // Öncelik ve durum güncelleme
  document.querySelectorAll('.priority-select, .status-select').forEach(select => {
    select.addEventListener('change', function() {
      const field = this.classList.contains('priority-select') ? 'priority' : 'status';
      fetch('update_status.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          id: this.dataset.id,
          field: field,
          value: this.value
        })
      }).then(() => {
        // Satırın sınıfını güncelle
        const row = this.closest('tr');
        if (field === 'priority') {
          row.className = row.className.replace(/priority-\d/g, '') + ` priority-${this.value}`;
        } else {
          row.className = row.className.replace(/status-\w+/g, '') + ` status-${this.value}`;
        }
      });
    });
  });

  // Ekip atama işlemi
  document.querySelectorAll('.assign-btn').forEach(btn => {
    btn.addEventListener('click', function() {
      const assignmentDiv = this.nextElementSibling;
      assignmentDiv.style.display = assignmentDiv.style.display === 'none' ? 'block' : 'none';
    });
  });

  document.querySelectorAll('.confirm-assign').forEach(btn => {
    btn.addEventListener('click', function() {
      const caseId = this.closest('td').querySelector('.assign-btn').dataset.id;
      const teamId = this.previousElementSibling.value;
      
      fetch('assign_team.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ caseId, teamId })
      }).then(() => location.reload());
    });
  });

  // Grafik oluşturma
  const ctx = document.getElementById('incidentChart').getContext('2d');
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: ['Deprem', 'Sel', 'Yangın', 'Diğer'],
      datasets: [{
        label: 'Acil Durum Türleri',
        data: [
          <?= $chart_data['deprem'] ?>,
          <?= $chart_data['sel'] ?>,
          <?= $chart_data['yangin'] ?>,
          <?= $chart_data['digery'] ?>
        ],
        backgroundColor: [
          '#e74c3c',
          '#3498db',
          '#f39c12',
          '#7f8c8d'
        ]
      }]
    },
    options: {
      responsive: true,
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });

  // Yardımcı fonksiyonlar
  function getPriorityLabel(priority) {
    const labels = {1: 'Düşük', 2: 'Orta', 3: 'Yüksek'};
    return labels[priority] || 'Bilinmiyor';
  }

  function getStatusLabel(status) {
    const labels = {'pending': 'Beklemede', 'assigned': 'İşleme Alındı', 'resolved': 'Çözüldü'};
    return labels[status] || 'Bilinmiyor';
  }

  // Service Worker kaydı
  if ('serviceWorker' in navigator) {
    navigator.serviceWorker.register('sw.js').then(registration => {
      console.log('ServiceWorker registration successful');
    });
  }

  // Gerçek zamanlı güncelleme
  let lastId = <?= !empty($locations) ? (int)$locations[0]['id'] : 0 ?>;

  if (Notification.permission !== "granted") {
    Notification.requestPermission();
  }

  setInterval(() => {
    fetch('check_new.php?last_id=' + lastId)
      .then(res => res.json())
      .then(data => {
        if (data.new && data.record) {
          lastId = data.record.id;
          if (Notification.permission === "granted") {
            new Notification("Yeni Acil Durum!", {
              body: `${data.record.name} ${data.record.surname} - ${data.record.emergency_type}`,
              icon: 'image/karkut.png'
            });
          }
          // Sayfayı yenile
          location.reload();
        }
      });
  }, 10000);
</script>
</body>
</html>

<?php
function getEmergencyTypeLabel($type) {
  $labels = [
    'deprem' => 'Deprem',
    'sel' => 'Sel',
    'yangin' => 'Yangın',
    'bina_yikilmasi' => 'Bina Yıkılması',
    'trafik_kazasi' => 'Trafik Kazası',
    'medikal' => 'Medikal Acil Durum',
    'digery' => 'Diğer'
  ];
  return $labels[$type] ?? 'Diğer';
}

function formatPhoneNumber($phone) {
  if (empty($phone)) return '-';
  $phone = preg_replace('/[^0-9]/', '', $phone);
  if (strlen($phone) === 10) {
    return substr($phone, 0, 3) . ' ' . substr($phone, 3, 3) . ' ' . substr($phone, 6, 2) . ' ' . substr($phone, 8, 2);
  }
  return $phone;
}

function getTeamName($pdo, $teamId) {
  if (empty($teamId)) return '-';
  $stmt = $pdo->prepare("SELECT name FROM ekipler WHERE id = ?");
  $stmt->execute([$teamId]);
  return $stmt->fetchColumn() ?: '-';
}
?>